/*:
 * @target MZ
 * @plugindesc v1.4.1 起動時初期化＋タイトル言語選択＋用語\V[n]展開（messages対応/通貨非干渉）＋タイトル4行固定＋タイトル→オプション/コンテニューでフォールバック
 * @author HS
 *
 * @help
 * 1) 起動時：OS言語→言語インデックスをConfig保持
 * 2) タイトル：言語メニュー追加。選択→変数へ反映→即リフレッシュ
 * 3) 用語：コマンド/基本/能力/メッセージで \V[n] 展開（タイトル・タイトル経由のオプション・ロード画面はフォールバック優先）
 * 4) タイトルのメインコマンドは常に4行（パラメータで変更可）
 * ※ currencyUnit は触りません（早期参照エラー回避）
 *
 * @param languageVarId
 * @text 言語インデックス変数ID
 * @type variable
 * @default 0
 *
 * @param languages
 * @text 言語一覧（表示名/コード/インデックス）
 * @type struct<Lang>[]
 * @default [{"name":"日本語","code":"ja","index":"0"},{"name":"English","code":"en","index":"1"},{"name":"简体中文","code":"zh-CN","index":"2"},{"name":"한국어","code":"ko","index":"3"}]
 *
 * @param titleCommandText
 * @text タイトル表示名
 * @type string
 * @default Language / 言語
 *
 * @param reserveCommonOnNewGame
 * @text 新規開始時に予約する共通EVID
 * @type common_event
 * @default 0
 *
 * @param titleFallbacks
 * @text タイトル用フォールバック
 * @type struct<TitleVar>[]
 * @default []
 *
 * @param titleVisibleRows
 * @text タイトルコマンドの表示行数
 * @type number
 * @min 1
 * @default 5
 */
/*~struct~Lang:
 * @param name
 * @text 表示名
 * @type string
 * @default 日本語
 * @param code
 * @text BCP47コード
 * @type string
 * @default ja
 * @param index
 * @text インデックス
 * @type number
 * @min 0
 * @default 0
 */
/*~struct~TitleVar:
 * @param varId
 * @text 変数ID
 * @type number
 * @min 1
 * @default 175
 * @param value0
 * @text 0:日本語
 * @type string
 * @default ニューゲーム
 * @param value1
 * @text 1:English
 * @type string
 * @default New Game
 * @param value2
 * @text 2:简体中文
 * @type string
 * @default 开始游戏
 * @param value3
 * @text 3:한국어
 * @type string
 * @default 뉴 게임
 * @param def
 * @text その他(Default)
 * @type string
 * @default
 */

(() => {
  "use strict";

  const PN = document.currentScript.src.match(/([^/]+)\.js$/)[1];
  const P  = PluginManager.parameters(PN);

  const VAR_ID      = Number(P.languageVarId || 0);
  const TITLE_TEXT  = String(P.titleCommandText || "Language / 言語");
  const TITLE_ROWS  = Math.max(1, Number(P.titleVisibleRows || 5));

  const parseJSON = (s, d)=>{ try{ return JSON.parse(s); }catch{ return d; } };

  const LANGS = (parseJSON(P.languages, []))
    .map(x => typeof x === "string" ? parseJSON(x, null) : x)
    .filter(o => o && o.name !== undefined && o.index !== undefined)
    .map(o => ({ name:String(o.name), code:String(o.code||""), index:Number(o.index) }));

  const TITLE_FB = {};
  for (const it of (parseJSON(P.titleFallbacks, []))
        .map(x => typeof x === "string" ? parseJSON(x, null) : x).filter(Boolean)) {
    const id = Number(it.varId||0); if (!id) continue;
    TITLE_FB[id] = { 0:it.value0??"", 1:it.value1??"", 2:it.value2??"", 3:it.value3??"", def:it.def??"" };
  }

  const CE_ON_NEW = Number(P.reserveCommonOnNewGame || 0);

  // -------- Config: language index --------
  const _setDefaults = ConfigManager.setDefaults;
  ConfigManager.setDefaults = function(){
    _setDefaults.call(this);
    this.hs_langIndex = detectByNavigator();
  };
  const _applyData = ConfigManager.applyData;
  ConfigManager.applyData = function(cfg){
    _applyData.call(this, cfg);
    if (typeof cfg.hs_langIndex === "number") this.hs_langIndex = cfg.hs_langIndex;
    if (this.hs_langIndex == null) this.hs_langIndex = detectByNavigator();
  };
  const _makeData = ConfigManager.makeData;
  ConfigManager.makeData = function(){
    const d = _makeData.call(this);
    d.hs_langIndex = this.hs_langIndex;
    return d;
  };

  function detectByNavigator(){
    const lang = (navigator.languages && navigator.languages[0]) || navigator.language || "en";
    const c = String(lang).toLowerCase();
    const exact = LANGS.find(l => String(l.code).toLowerCase() === c);
    if (exact) return exact.index;
    const pref  = LANGS.find(l => String(l.code).toLowerCase() === c.split("-")[0]);
    if (pref) return pref.index;
    const en = LANGS.find(l => String(l.code).toLowerCase() === "en");
    return en ? en.index : (LANGS[0]?.index ?? 0);
  }

  // -------- Reflect to variable / common event --------
  function applyLangIndex(idx, save=true){
    ConfigManager.hs_langIndex = Number(idx);
    if (save) ConfigManager.save();
    if (VAR_ID>0 && window.$gameVariables) $gameVariables.setValue(VAR_ID, Number(idx));
  }

  const _setupNewGame = DataManager.setupNewGame;
  DataManager.setupNewGame = function(){
    _setupNewGame.call(this);
    applyLangIndex(ConfigManager.hs_langIndex, false);
    if (CE_ON_NEW>0) $gameTemp.reserveCommonEvent(CE_ON_NEW);
  };

  const _loadGame = DataManager.loadGame;
  DataManager.loadGame = async function(id){
    const ok = await _loadGame.call(this, id);
    if (ok) applyLangIndex(ConfigManager.hs_langIndex, false);
    return ok;
  };

  // -------- Title: add "Language" & 4-row fix --------
  const _makeCommandList = Window_TitleCommand.prototype.makeCommandList;
  Window_TitleCommand.prototype.makeCommandList = function(){
    _makeCommandList.call(this);
    this.addCommand(TITLE_TEXT, "hs_lang");
  };

  Window_TitleCommand.prototype.numVisibleRows = function(){
    const items = (typeof this.maxItems==="function") ? this.maxItems() : 4;
    return Math.min(TITLE_ROWS, items);
  };

  const _createCommandWindow = Scene_Title.prototype.createCommandWindow;
  Scene_Title.prototype.createCommandWindow = function(){
    _createCommandWindow.call(this);

    this._commandWindow.setHandler("hs_lang", this.commandHsLang.bind(this));

    // 4行ぶんの高さに強制リサイズ
    const x = this._commandWindow.x;
    const y = this._commandWindow.y;
    const w = this._commandWindow.width;
    const h = this._commandWindow.fittingHeight(TITLE_ROWS);
    this._commandWindow.move(x, y, w, h);

    // タイトルからのオプション遷移をフラグ化
    const origOpt = this._commandWindow._handlers && this._commandWindow._handlers["options"];
    if (origOpt) {
      this._commandWindow.setHandler("options", () => {
        window.HS_TitleInvokedOptions = true;
        origOpt();
      });
    }

    this.createHsLangWindow();
  };

  // 互換API
  Scene_Title.prototype.mainCommandRows   = function(){ return TITLE_ROWS; };
  Scene_Title.prototype.commandWindowRows = function(){ return TITLE_ROWS; };

  // -------- Language selection window --------
  Scene_Title.prototype.createHsLangWindow = function(){
    const ww = 420, wh = Math.min(56 + LANGS.length * 36, 360);
    const wx = (Graphics.boxWidth - ww)/2;
    const wy = (Graphics.boxHeight - wh)/2;
    this._hsLangWindow = new Window_HS_LangList(new Rectangle(wx, wy, ww, wh));
    this._hsLangWindow.setHandler("ok", this.onHsLangOk.bind(this));
    this._hsLangWindow.setHandler("cancel", this.onHsLangCancel.bind(this));
    this._hsLangWindow.openness = 0;
    this.addWindow(this._hsLangWindow);
  };
  Scene_Title.prototype.commandHsLang = function(){
    this._commandWindow.deactivate();
    const pos = LANGS.findIndex(l=>l.index===ConfigManager.hs_langIndex);
    this._hsLangWindow.select(Math.max(0,pos));
    this._hsLangWindow.open();
    this._hsLangWindow.activate();
  };
  Scene_Title.prototype.onHsLangOk = function(){
    const idx = this._hsLangWindow.currentExt();
    SoundManager.playOk();
    applyLangIndex(idx, true);
    this._hsLangWindow.close();
    if (this._commandWindow?.refresh) this._commandWindow.refresh();
    this._commandWindow.activate();
  };
  Scene_Title.prototype.onHsLangCancel = function(){
    SoundManager.playCancel();
    this._hsLangWindow.close();
    this._commandWindow.activate();
  };

  function Window_HS_LangList(rect){ Window_Command.prototype.initialize.call(this, rect); this.refresh(); }
  Window_HS_LangList.prototype = Object.create(Window_Command.prototype);
  Window_HS_LangList.prototype.constructor = Window_HS_LangList;
  Window_HS_LangList.prototype.makeCommandList = function(){
    for (const l of LANGS) this.addCommand(String(l.name), "ok", true, Number(l.index));
  };
  Window_HS_LangList.prototype.drawItem = function(i){
    const r=this.itemLineRect(i), it=this._list[i];
    const name=(it&&typeof it.name==="string")?it.name:String(it?.ext??"");
    this.resetTextColor(); this.changePaintOpacity(this.isCommandEnabled(i));
    this.drawText(name, r.x, r.y, r.width);
  };

  // -------- Replace \V[n] --------
  const langIndex = ()=>Number(ConfigManager.hs_langIndex ?? 0);
  const fbValue = (id)=>{
    const t=TITLE_FB[id]; if(!t) return "";
    const li=langIndex(); const hit=(li in t)?t[li]:t.def;
    return String(hit ?? "");
  };

  // タイトル/ロード/タイトル経由のオプションはフォールバック優先
  function inPreGameScene(){
    const s = SceneManager._scene;
    if (s instanceof Scene_Title) return true;
    if (s instanceof Scene_Load) return true;       // コンテニュー（ロード）
    if (s instanceof Scene_Options && window.HS_TitleInvokedOptions) return true;
    if (s instanceof Scene_GameEnd) return true;    // 任意：ゲーム終了系
    return false;
  }

  function replaceVars(text){
    if(!text) return "";
    return text.replace(/\\V\[(\d+)\]/gi,(_,n)=>{
      const id=Number(n);
      if (inPreGameScene()) {
        const fb=fbValue(id);
        return fb || "";
      }
      if (window.$gameVariables && $gameVariables._data) {
        const v=$gameVariables.value(id);
        return String(v ?? "");
      }
      return "";
    }).replace(/\\\\/g,"\\");
  }

  // TextManager hooks
  const _tmCommand = TextManager.command;
  if (typeof _tmCommand === "function") {
    TextManager.command = function(){ try{ return replaceVars(_tmCommand.apply(this,arguments)); }catch{ return ""; } };
  }
  const _tmMessage = TextManager.message;
  if (typeof _tmMessage === "function") {
    TextManager.message = function(){ try{ return replaceVars(_tmMessage.apply(this,arguments)); }catch{ return ""; } };
  }

  // DataManager onLoad: wrap $dataSystem.terms
  const _onLoad = DataManager.onLoad;
  DataManager.onLoad = function(obj){
    _onLoad.call(this,obj);
    if (obj === $dataSystem && $dataSystem.terms) {
      const t=$dataSystem.terms;
      const arrs=[t.commands,t.params,t.basic];
      for (const arr of arrs) {
        if (Array.isArray(arr)) for (let i=0;i<arr.length;i++) wrapGetter(arr,i);
      }
      if (t.messages && typeof t.messages === "object") {
        for (const k of Object.keys(t.messages)) wrapGetter(t.messages,k);
      }
      // currencyUnit は触らない
    }
  };
  function wrapGetter(obj,key){
    if(!obj || !Object.prototype.hasOwnProperty.call(obj,key)) return;
    let v=obj[key];
    Object.defineProperty(obj,key,{
      get(){ return replaceVars(v); },
      set(nv){ v=nv; },
      configurable:true
    });
  }

  // Scene_Options 終了時にフラグ解除
  const _optTerminate = Scene_Options.prototype.terminate;
  Scene_Options.prototype.terminate = function(){
    _optTerminate.call(this);
    if (window.HS_TitleInvokedOptions) window.HS_TitleInvokedOptions = false;
  };

  // 起動時：Configの言語を変数に反映
  const _start = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function(){
    _start.call(this);
    if (VAR_ID>0 && window.$gameVariables) $gameVariables.setValue(VAR_ID, Number(ConfigManager.hs_langIndex));
  };
// === 強制同期ミニパッチ（ロード/マップ開始でも念押し） ===
(() => {
  const PRM = PluginManager.parameters("HS_LangSimple");
  const VAR_ID = Number(PRM.languageVarId || 0);

  function syncLangVar() {
    if (VAR_ID > 0 && window.$gameVariables) {
      $gameVariables.setValue(VAR_ID, Number(ConfigManager.hs_langIndex ?? 0));
    }
  }

  // ロード成功直後にもう一度、変数へ反映
  const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
  Scene_Load.prototype.onLoadSuccess = function() {
    _onLoadSuccess.call(this);
    syncLangVar();
  };

  // 念のため、マップ開始時（初回フレーム）にも反映
  const _mapStart = Scene_Map.prototype.start;
  Scene_Map.prototype.start = function() {
    _mapStart.call(this);
    syncLangVar();
  };
})();

})();
